/*
 * This file is part of cchttpd.
 * Copyright (C) 2012-2014  Guillaume Quintin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#define _POSIX_C_SOURCE 200112L /* not happy with that, should be 1 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <arpa/inet.h>
#include <netinet/in.h>

/* some defines */
#define VERSION    "cchttpd version 0.5"
#define LEN        (8 * 1024)
#define BUF_LEN    (4 * 1024)
#define BF_LEN     (LEN + INET6_ADDRSTRLEN + 1)
#define DEV_NULL   "/dev/null"
#define MAX(a,b)   ( (a) > (b) ? (a) : (b) )
#define V_ERR       0
#define V_LOG_SVR   1
#define V_LOG_CONN  5
#define V_DEBUG     10

/* global variables */
typedef struct conn conn;
struct conn {
  FILE *in,*out,*err;
  char ip[INET6_ADDRSTRLEN];
};

char *home;
int argc2;
char **argv2,*ip;
char hostname[BUF_LEN];
int daemonize,verbose,as_server;
int port;
int sigpipe[2];
int children;
conn main_conn;
uid_t uid;
gid_t gid;
int read_timeout;

/* function prototypes */
#include "cchttpd.h"

/* the actual code */
#include "init.c"
#include "mime.c"
#include "send.c"
#include "mycgi.c"
#include "get_method.c"
#include "signals.c"
#include "socket.c"
#include "logger.c"

const char *relative_path(const char *s) {
  if ( s[0] == '/' ) return &s[1];
  return s;
}

void exit2(int status,conn *c) {
  logger(V_DEBUG,c,"exit with status %d",status);
  if ( c->in ) fclose(c->in);
  if ( c->out ) fclose(c->out);
  if ( c->err ) fclose(c->err);
  exit(status);
}

void assert2(int cond,const char *s) {
  if ( !cond ) {
    fprintf(stderr,"cchttpd: %s\n",s);
    perror("cchttpd");
    exit(-1);
  }
}

void proceed(int fdin,int fdout,int fderr,const char *ip) {
  conn c;
  struct timeval tv;

  c.in = NULL;
  c.out = NULL;
  c.err = fdopen(fderr,"w");
  strcpy(c.ip,ip);
  if ( c.err == NULL ) {
    /* Well, almost nothing we can do about that
       we exit we code -2 */
    close(fdin);
    if ( fdin != fdout ) close(fdout);
    close(fderr);
    exit(-2);
  }
  
  /* From here we have, a priori, a valid err fd */
 
  /* set a timeout of 5 seconds for incomming connection */
  tv.tv_sec = 5;
  tv.tv_usec = 0;
  if ( fdin != 0 && setsockopt(fdin,SOL_SOCKET,SO_RCVTIMEO,
                               (void*)&tv,sizeof(tv)) )
  {
    close(fdin);
    logger(V_ERR,&c,"cannot set timeout for %d, this is weird!",fdin);
    exit2(EXIT_FAILURE,&c);
  }

  /* create two distinct fds for read and write */
  if ( fdin == fdout ) {
    fdout = dup(fdout);
    if ( fdout == -1 ) {
      close(fdin);
      logger(V_ERR,&c,"cannot dup(%d), this is weird!",fdin);
      exit2(EXIT_FAILURE,&c);
    }
  }

  /* wrap up low-level fds (int's) into (FILE*)'s */
  c.in = fdopen(fdin,"r");
  c.out = fdopen(fdout,"w");
  if ( c.in == NULL || c.out == NULL ) {
    logger(V_ERR,&c,"cannot fdopen, this is weird!",fdin);
    exit2(EXIT_FAILURE,&c);
  }
  logger(V_DEBUG,&c,"chdir to %s",home);
  if ( chdir(home) ) {
    send_internal_error(&c);
    logger(V_ERR,&c,"cannot chdir into %s",home);
    exit2(EXIT_SUCCESS,&c);
  }

  /* treat the client request */
  check_get_method(&c);
  do_get_method(&c);
  exit2(EXIT_SUCCESS,&c);
}

int main(int argc,char **argv) {
  if ( argc == 2 && !strcmp(argv[1],"--version") ) {
    puts(VERSION);
    return 0;
  }
  init(argc,argv);
  if ( as_server ) {
    daemon_main_loop();
  }
  else {
    proceed(0,1,2,"[stdin]");
  }
  return 0;
}

