/*
 * This file is part of cchttpd.
 * Copyright (C) 2012-2014  Guillaume Quintin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

void send_exit_conn_closed(int ret,conn *c) {
  logger(V_LOG_CONN,c,
         "unexpected error while writing into file descriptor");
  logger(V_LOG_CONN,c,"connection closed by client?");
  exit2(ret,c);
}

void fflush2(conn *c,FILE *out) {
  if ( fflush(out) == EOF ) send_exit_conn_closed(EXIT_SUCCESS,c);
}

void fputs2(const char *s,FILE *out,conn *c) {
  if ( fputs(s,out) == EOF ) send_exit_conn_closed(EXIT_SUCCESS,c);
}

void fprintf2(conn *c,FILE *out,const char *format,...) {
  va_list ap;
  va_start(ap,format);
  if ( vfprintf(out,format,ap) < 0 ) {
    va_end(ap);
    send_exit_conn_closed(EXIT_SUCCESS,c);
  }
  va_end(ap);
}

/********************************************************************
                               Header
********************************************************************/

void send_headers(conn *c,const char *status_code,
                          const char *reason_phrase,
                          const char *mime_type,
                          unsigned long size)
{
  FILE *out;
  out = c->out;
  fprintf2(c,out,"HTTP/1.1 %s %s\r\n"
                 "Server: cchttpd\r\n"
                 "Content-Type: %s\r\n"
                 "Content-Length: %ld\r\n"
                 "Connection: close\r\n\r\n",
                 status_code,reason_phrase,mime_type,size);
}

/********************************************************************
                             Redirection
********************************************************************/

void send_redirect(conn *c,const char *where) {
  FILE *out;
  out = c->out;
  fprintf2(c,out,"HTTP/1.1 301 Moved Permanently\r\n"
                 "Server: cchttpd\r\n"
                 "Location: %s\r\n"
                 "Connection: close\r\n\r\n",where);
}

/********************************************************************
                            Error sending
********************************************************************/

/* errors */
#define BAD_REQUEST \
    "400 Bad Request: unable to understand request.\n" VERSION ".\n"

#define INTERNAL_SERVER_ERROR \
    "500 Internal Server Error.\n" VERSION ".\n"

#define URI_TOO_LONG \
    "414 Request-URI Too Long.\n" VERSION ".\n"

#define NOT_FOUND \
    "404 Not Found: resource not found on server.\n" VERSION ".\n"

#define NOT_IMPLEMENTED \
    "501 Not Implemented: only GET method is supported.\n" VERSION ".\n"

void send_not_implemented(conn *c) {
  FILE *out;
  out = c->out;
  logger(V_LOG_CONN,c,"sending error 501 (not implemented)");
  send_headers(c,"501","Not Implemented","text/plain",
               sizeof(NOT_IMPLEMENTED));
  fputs2(NOT_IMPLEMENTED,out,c);
}

void send_not_found(conn *c) {
  FILE *out;
  out = c->out;
  logger(V_LOG_CONN,c,"sending error 404 (not found)");
  send_headers(c,"404","Not Found","text/plain",sizeof(NOT_FOUND));
  fputs2(NOT_FOUND,out,c);
}

void send_bad_request(conn *c) {
  FILE *out;
  out = c->out;
  logger(V_LOG_CONN,c,"sending error 400 (bad request)");
  send_headers(c,"400","Bad request","text/plain",
               sizeof(BAD_REQUEST));
  fputs2(BAD_REQUEST,out,c);
}

void send_internal_error(conn *c) {
  FILE *out;
  out = c->out;
  logger(V_LOG_CONN,c,"sending error 500 (internal error)");
  send_headers(c,"500","Internal Server Error","text/plain",
               sizeof(INTERNAL_SERVER_ERROR));
  fputs2(INTERNAL_SERVER_ERROR,out,c);
}

void send_uri_too_long(conn *c) {
  FILE *out;
  out = c->out;
  logger(V_LOG_CONN,c,"sending error 414 (URI too long)");
  send_headers(c,"414","Request-URI Too Long","text/plain",
               sizeof(URI_TOO_LONG));
  fputs2(URI_TOO_LONG,out,c);
}


/********************************************************************
                            File sending
********************************************************************/

/* When this function is called empty_input() has been called */
void try_send_file(conn *c,const char *file) {
  char buf[BUF_LEN];
  FILE *in,*out;
  long sz;
  out = c->out;
  logger(V_DEBUG,c,"sending %s",file);
  in = fopen(relative_path(file),"r");
  if ( !in ) {
    logger(V_DEBUG,c,"cannot open %s",file);
    return;
  }
  if ( fseek(in,0,SEEK_END) ) {
    logger(V_DEBUG,c,"cannot determine the size of %s",file);
    fclose(in);
    exit2(EXIT_SUCCESS,c);
  }
  sz = ftell(in);
  if ( sz == -1 ) {
    fclose(in);
    return;
  }
  if ( fseek(in,0,SEEK_SET) ) {
    fclose(in);
    return;
  }
  send_headers(c,"200","OK",get_mime_type(file),sz);
  for(;;) {
    unsigned long len = fread(buf,1,BUF_LEN,in);
    if ( len == 0 ) break;
    if ( fwrite(buf,len,1,out) <= 0 ) send_exit_conn_closed(0,c);
  }
  fclose(in);
  fflush2(c,out);
  logger(V_DEBUG,c,"OK, %s sent",file);
  exit2(EXIT_SUCCESS,c);
}

/********************************************************************
                         Directory listing
********************************************************************/

unsigned long sz_kb(unsigned long sz) {
  unsigned long kb;
  kb = sz / 1024;
  if ( (sz % 1024) != 0 ) kb++;
  return kb;
}

unsigned long log10_sz_kb(unsigned long sz) {
  int i;
  unsigned long kb;
  kb = sz_kb(sz);
  if ( kb == 0 ) return 1;
  for( i = 0 ; kb > 0 ; kb /= 10,i++ );
  return i;
}

#define LS_DIR_BEGIN_1   "<!DOCTYPE HTML PUBLIC " \
                                   "\"-//W3C//DTD HTML 4.01//EN\"\n" \
                         "    \"http://www.w3.org/TR/html4/strict.dtd\">\n" \
                         "<html>\n" \
                         "<head>\n" \
                         "  <title>Index of "
#define LS_DIR_BEGIN_2     "</title>\n" \
                         "</head>\n" \
                         "<body>\n" \
                         "  <p><b>Index of "
#define LS_DIR_BEGIN_3     "</b></p>\n" \
                         "  <hr>\n" \
                         "  <table border=\"0\">\n" \
                         "    <tr><td><b>Name</b></td>" \
                                 "<td><b>Size</b></td></tr>\n"
#define LS_DIR_ENTRY_1   "    <tr><td><a href=\""
#define LS_DIR_ENTRY_2       "\">"
#define LS_DIR_ENTRY_3       "</a></td><td>"
#define LS_DIR_ENTRY_4       " KB<td></tr>\n"
#define LS_DIR_END       "  </table>\n" \
                         "</body></html>"

/* When this function is called empty_input() has been called */
void try_list_dir(conn *c,const char *dir) {
  DIR *dd;
  FILE *out;
  struct dirent *entry;
  struct stat buf;
  unsigned long sz;

  if ( dir[0] != '/' || dir[1] != 0 ) {
    if ( chdir(relative_path(dir)) ) return;
  }

  /* First, we determine the size of the data */
  dd = opendir(".");
  if ( !dd ) return;
  sz = 0;
  for( ; (entry = readdir(dd)) ; ) {
    if ( entry->d_name[0] == '.' ) continue;
    if ( !stat(entry->d_name,&buf) ) {
      sz += 2 * strlen(entry->d_name); /* 2x the name of the entry */
      sz += log10_sz_kb(buf.st_size); /* the size of the entry */
      sz += sizeof(LS_DIR_ENTRY_1) - 1 + /* the decorating HTML */
            sizeof(LS_DIR_ENTRY_2) - 1 +
            sizeof(LS_DIR_ENTRY_3) - 1 +
            sizeof(LS_DIR_ENTRY_4) - 1;
      /* a dir has an extra '/' at the end, two times */
      if ( S_ISDIR(buf.st_mode) ) sz += 2;
    }
  }
  if ( closedir(dd) ) exit2(EXIT_FAILURE,c);
  sz += 2 * strlen(dir) + /* 2x the name of the dir */
        sizeof(LS_DIR_BEGIN_1) - 1 + /* the decorating HTML */
        sizeof(LS_DIR_BEGIN_2) - 1 +
        sizeof(LS_DIR_BEGIN_3) - 1 +
        sizeof(LS_DIR_END) - 1;

  /* Then, we send the data */
  dd = opendir(".");
  if ( !dd ) return;
  out = c->out;
  send_headers(c,"200","OK","text/html",sz);
  fputs2(LS_DIR_BEGIN_1,out,c);
  fputs2(dir,out,c);
  fputs2(LS_DIR_BEGIN_2,out,c);
  fputs2(dir,out,c);
  fputs2(LS_DIR_BEGIN_3,out,c);
  for( ; (entry = readdir(dd)) ; ) {
    if ( entry->d_name[0] == '.' ) continue;
    if ( !stat(entry->d_name,&buf) ) {
      fputs2(LS_DIR_ENTRY_1,out,c);
      fputs2(entry->d_name,out,c);
      if ( S_ISDIR(buf.st_mode) ) fputs2("/",out,c);
      fputs2(LS_DIR_ENTRY_2,out,c);
      fputs2(entry->d_name,out,c);
      if ( S_ISDIR(buf.st_mode) ) fputs2("/",out,c);
      fputs2(LS_DIR_ENTRY_3,out,c);
      if ( fprintf(out,"%lu",sz_kb(buf.st_size)) <= 0 ) {
        send_exit_conn_closed(0,c);
      }
      fputs2(LS_DIR_ENTRY_4,out,c);
    }
  }
  fputs2(LS_DIR_END,out,c);
  fflush2(c,out);
  closedir(dd);
  exit2(EXIT_SUCCESS,c);
}

