/*
 * This file is part of cchttpd.
 * Copyright (C) 2013-2014  Guillaume Quintin.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

void do_daemonize(void) {
  pid_t pid1;
  pid1 = fork();
  assert2((pid1 >= 0),"cannot fork the child");
  if ( pid1 > 0 ) {
    waitpid(pid1,NULL,0);
    exit(EXIT_SUCCESS);
  }
  { /* in the child */
    pid_t pid2;
    pid2 = fork();
    assert2((pid2 >= 0),"cannot fork the grandchild");
    if ( pid2 == 0 ) return; /* in the grandchild */
    exit(EXIT_SUCCESS);
  }
}

void accept_connection(int s,int ns,const char *ip) {
  pid_t pid;
  int err1,err2;

  pid = fork();
  if ( pid == -1 ) {
    logger(V_ERR,&main_conn,"cannot fork to serve client");
    close(ns);
    return;
  }
  if ( pid ) {
    logger(V_DEBUG,&main_conn,"birth of my children %lu",pid);
    children++;
    return;
  }
  /* from here we are in the child, no error checking */
  err1 = sig_default();
  err2 = 0;
  err2 = ( close(s) == -1 ||
           close(sigpipe[0]) == -1 ||
           close(sigpipe[1]) == -1 );
  if ( !err1 && !err2 ) {
    proceed(ns,ns,fileno(main_conn.err),ip);
    /* process exists here */
  }
  if ( err1 ) {
    logger(V_ERR,&main_conn,
           "cannot default handlers for all signals");
  }
  if ( err2 ) {
    logger(V_ERR,&main_conn,"cannot close some fds");
  }
  close(ns);
  exit(EXIT_SUCCESS);
}

void daemon_main_loop(void) {
  int s,ret,i,n;
  struct sockaddr_in sin;
 
  ret = EXIT_SUCCESS;

  /* create the socket */
  s = socket(AF_INET,SOCK_STREAM,0);
  assert2((s >= 0),"Cannot create socket");
  memset(&sin,0,sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_port = htons(port);
  if ( !strcmp(ip,"0.0.0.0") ) {
    sin.sin_addr.s_addr = INADDR_ANY;
  }
  else {
    sin.sin_addr.s_addr = inet_addr(ip);
  }
  if ( bind(s,(struct sockaddr*)(&sin),sizeof(sin)) ) {
    close(s);
    fprintf(stderr,"cchttpd: cannot bind socket to address %s:%d\n",
            ip,port);
    perror("cchttpd");
    exit(EXIT_FAILURE);
  }
  errno = 0;
  if ( listen(s,5) ) {
    close(s);
    fprintf(stderr,"cchttpd: cannot listen with socket %d\n",s);
    perror("cchttpd");
    exit(-1);
  }
  logger(V_LOG_SVR,&main_conn,"server started on %s:%d",ip,port);
  
  /* privilege dropdown */
  assert2((setgid(gid) == 0),"cannot change gid");
  assert2((setuid(uid) == 0),"cannot change uid");

  /* daemonize cchttpd */
  if ( daemonize ) do_daemonize();

  /* signals stuff */
  assert2((pipe(sigpipe) == 0),"cannot create pipe for signals");
  init_signals();
  
  /* main loop here */
  while(1) {
    fd_set rfds;
    int ns,r;

    /* select stuff */
    FD_ZERO(&rfds);
    FD_SET(s,&rfds);
    FD_SET(sigpipe[0],&rfds);
    errno = 0;
    r = select(MAX(0,sigpipe[0]) + 1,&rfds,0,0,0);
    if ( r == -1 ) {
      if ( errno == EINTR ) continue;
      logger(V_ERR,&main_conn,"error %d in select: %s",
             errno,strerror(errno));
      break;
    }
    
    /* signal received */
    if ( FD_ISSET(sigpipe[0],&rfds) ) {
      char sig;
      read(sigpipe[0],&sig,1);
      if ( sig == 't' ) {
        logger(V_LOG_SVR,&main_conn,"SIGTERM or SIGINT received");
        break;
      }
      if ( sig == 'c' ) {
        pid_t pid;
        pid = waitpid(-1,NULL,0);
        if ( pid == -1 ) {
          logger(V_ERR,&main_conn,"waitpid failed, weird!");
        }
        else {
          logger(V_DEBUG,&main_conn,"child %lu has terminated",pid);
          children--;
        }
      }
    }

    /* incoming connection */
    if ( FD_ISSET(s,&rfds) ) {
      socklen_t sz_nsin;
      struct sockaddr_in nsin;
      char ip[INET6_ADDRSTRLEN];
      int nport;
      
      memset(&nsin,0,sizeof(sin));
      sz_nsin = sizeof(nsin);
      ns = accept(s,(struct sockaddr*)(&nsin),&sz_nsin);
      if ( ns == -1 ) {
        logger(V_ERR,&main_conn,"cannot accept incoming connexion");
        continue;
      }
      if ( inet_ntop(nsin.sin_family,&nsin.sin_addr,ip,19) == NULL ) {
        strcpy(ip,"[unknown ip]");
      }
      nport = ntohs(nsin.sin_port);
      logger(V_DEBUG,&main_conn,
             "incoming connexion from %s:%d",ip,nport);
      accept_connection(s,ns,ip);
      close(ns);
    }
  }

  /* we close all open fds */
  sig_default();
  if ( children > 0 ) {
    sigset_t set;
    sigfillset(&set);
    sigdelset(&set,SIGCHLD);
    sigdelset(&set,SIGCONT);
    if ( sigprocmask(SIG_BLOCK,&set,NULL) ) {
      logger(V_ERR,&main_conn,
             "failed to block signals for closing all fds,"
             " continuing anyway");
    }
    logger(V_LOG_SVR,&main_conn,
           "still have children, sending SIGTERM");
    kill(0,SIGTERM);
    n = children;
    for( i = 0 ; i < n ; i++ ) {
      if ( waitpid(-1,NULL,WNOHANG) ) children--;
    }
    if ( children > 0 ) {
      logger(V_LOG_SVR,&main_conn,
             "still have children, sending SIGKILL this time");
      kill(0,SIGKILL);
      n = children;
      for( i = 0 ; i < n ; i++ ) {
        if ( waitpid(-1,NULL,WNOHANG) ) children--;
      }
      if ( children > 0 ) {
        logger(V_LOG_SVR,&main_conn,
               "still have children, don't know what to do");
        ret = EXIT_FAILURE;
      }
    }
  }
  logger(V_LOG_SVR,&main_conn,"closing all fds");
  close(sigpipe[0]);
  close(sigpipe[1]);
  close(s);
  fclose(main_conn.err);
  exit(ret);
}

