/*
 * This file is part of imunal
 * Copyright 2013  Guillaume Quintin, Olivier Ruatta, Philippe Gaborit
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#define _POSIX_C_SOURCE 1

#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <imunal.h>

void print_err_nvar(const char *filename) {
  fprintf(stderr,"imunal: %s: number of variables incorrect\n",
          filename);
  fprintf(stderr,"imunal: %s: must be between 1 and %ld\n",
          filename,(long)IMUNAL_NVAR_MAX);
}

void print_ai(const char *filename,FILE *in,int verbose) {
  uma ai;
  struct bool_func *f;
  struct bool_func_ai_stats t;

  f = bool_func_read(in);
  if ( !f ) {
    fprintf(stderr,"imunal: %s: cannot read boolean function\n",
            filename);
    return;
  }
  if ( f->nvar < 0 || f->nvar > IMUNAL_NVAR_MAX ) {
    print_err_nvar(filename);
    return;
  }
  if ( verbose >= 2 ) {
    bool_func_print(stdout,f);
    fputc('\n',stdout);
  }
  ai = bool_func_get_ai(&t,f,IMUNAL_GRLEX);
  bool_func_free(f);
  fprintf(stdout,"%s: ",filename);
  if ( verbose >= 1 ) {
    double tt1,tt2;
    tt1 = (double)t.fiber / CLOCKS_PER_SEC;
    tt2 = (double)t.ai / CLOCKS_PER_SEC;
    fprintf(stdout,"%e, %e, ",tt1,tt2);
  }
  fprintf(stdout,"%ld\n",(long)ai);
  fflush(stdout);
}

void print_ai_std_function(uma nvar,int verbose,
                           struct points *(*func)(int),
                           const char *func_name)
{
  uma ai;
  struct points *pt;
  struct bool_func_ai_stats t;
  clock_t start,end;

  if ( nvar < 1 || nvar > IMUNAL_NVAR_MAX ) {
    print_err_nvar("majority_function");
    return;
  }
  start = clock();
  pt = func(nvar);
  if ( !pt ) {
    fprintf(stderr,"imunal: %s function: cannot compute "
                   "%s function\n",func_name,func_name);
    return;
  }
  end = clock();
  ai = bool_func_get_ai_pts(&t,pt,IMUNAL_GRLEX);
  points_free(pt);
  fprintf(stdout,"%s function in %ld variables: ",
          func_name,(long)nvar);
  if ( verbose >= 1 ) {
    double tt1,tt2;
    tt1 = (double)(end - start) / CLOCKS_PER_SEC;
    tt2 = (double)t.ai / CLOCKS_PER_SEC;
    fprintf(stdout,"%e, %e, ",tt1,tt2);
  }
  fprintf(stdout,"%ld\n",(long)ai);
  fflush(stdout);
}

int main(int argc,char **argv) {
  int i,verbose;
  if ( argc == 2 && !strcmp(argv[1],"--version") ) {
    fputs("imunal version 0.1\n",stdout);
    fflush(stdout);
    return 0;
  }
  verbose = 0;
  if ( argc == 1 ) {
    print_ai("stdin",stdin,verbose);
    return 0;
  }
  for( i = 1 ; i < argc ; i++ ) {
    FILE *in;
    if ( !strcmp(argv[i],"-v" ) ) {
      verbose = 1;
      continue;
    }
    else if ( !strcmp(argv[i],"-vv") ) {
      verbose = 2;
      continue;
    }
    else if ( !strcmp(argv[i],"-q") ) {
      verbose = 0;
      continue;
    }
    else if ( !strcmp(argv[i],"-") ) {
      print_ai("stdin",stdin,verbose);
      continue;
    }
    else if ( !strcmp(argv[i],"-majority") ) {
      uma nvar;
      i++;
      if ( argc - 1 < i ) {
        fputs("imunal: majority function: number of variables missing\n",
              stderr);
        
      }
      nvar = (uma)atol(argv[i]);
      print_ai_std_function(nvar,verbose,majority_function,"majority");
      continue;
    }
    else if ( !strcmp(argv[i],"-hwbf") ) {
      uma nvar;
      i++;
      if ( argc - 1 < i ) {
        fputs("imunal: hwbf function: number of variables missing\n",
              stderr);
        
      }
      nvar = (uma)atol(argv[i]);
      print_ai_std_function(nvar,verbose,hwbf_function,"hwbf");
      continue;
    }
    else {
      in = fopen(argv[i],"r");
      if ( !in ) continue;
      print_ai(argv[i],in,verbose);
      fclose(in);
      continue;
    }
  }
  return 0;
}
