/*
 * This file is part of imunal
 * Copyright 2013  Guillaume Quintin, Olivier Ruatta, Philippe Gaborit
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <stdlib.h>
#include <imunal.h>
#include <ram.h>

/* some useful stuff */
typedef unsigned WORD affine;

#define __SZ(a)      ((uma)(sizeof(a) * CHAR_BIT))
#define __MONOMIAL   __SZ(monomial)
#define MAX(a,b)     ( (a) > (b) ? (a) : (b) )
#define MIN(a,b)     ( (a) < (b) ? (a) : (b) )
#define MASK(n)      ((one << n) - one)
#define NaV          ((bit)-1)

PREFIX const unsigned WORD one = 1u;
PREFIX const unsigned WORD zero = 0u;
PREFIX const uma noid = -1;

/* functions prototypes */
  /* libimunal.c */
UNUSED PREFIX void tuple_print(FILE*,tuple,uma);
UNUSED PREFIX void uma_print(FILE*,uma*,uma);
PREFIX uma sum(uma);
PREFIX void points_sort(struct points*,int);
PREFIX uma main_algo_ex(struct points*,int,int,unsigned long);
PREFIX uma main_algo(struct points*,int);
  /* memory.c */
PREFIX void *default_malloc(size_t);
PREFIX void *default_realloc(void*,size_t);
PREFIX void *default_calloc(size_t,size_t);
PREFIX void default_free(void*);
PREFIX void *adjust2(void*,size_t);
PREFIX void *memdup2(void*,size_t);
  /* monomial.c */
PREFIX void monomial_print(FILE*,monomial);
PREFIX uma monomial_read(monomial*,FILE*);
PREFIX uma hamming(monomial);
PREFIX monomial monomial_next_grlex(monomial,uma);
PREFIX monomial monomial_next(monomial,uma,int);
PREFIX void monomial_quo(uma*,monomial*,monomial,uma);
PREFIX bit monomial_eval(monomial,tuple);

/* internal global variables */
PREFIX void *(*malloc2)(size_t) = default_malloc;
PREFIX void *(*realloc2)(void*,size_t) = default_realloc;
PREFIX void *(*calloc2)(size_t,size_t) = default_calloc;
PREFIX void (*free2)(void*) = default_free;

/* the code that does something */
#include <memory.c>
#include <monomial.c>
#include <bool_func.c>
#include <majority_function.c>
#include <hwbf_function.c>
#include <naive_version/naive_version.c>
#include <greedy_version/greedy_version.c>
#include <noswap_version/noswap_version.c>

/* misc functions below */
UNUSED PREFIX void tuple_print(FILE *out,tuple a,uma nvar) {
  uma i;
  for( i = 0 ; i < nvar ; i++ ) {
    fputc((int)((a >> i) & one) + '0',out);
  }
}

UNUSED PREFIX void uma_print(FILE *out,uma *a,uma na) {
  uma i;
 
  if ( a == NULL || na == 0 ) {
    fputs("NULL",stderr);
    return;
  }
  fprintf(out,"%ld",(long)a[0]);
  for( i = 1 ; i < na ; i++ ) {
    fprintf(out," %ld",(long)a[i]);
  }
}

void points_free(struct points *pts) {
  if ( pts->pt ) free2(pts->pt);
  free2(pts);
}

PREFIX uma sum(uma n) {
  return (n * (n + 1)) / 2;
}

PREFIX void points_sort(struct points *pts,int order) {
  typedef int (*cmp_t)(const void*,const void*);
  cmp_t cmp;
  switch(order) {
    case IMUNAL_GRLEX:
    default:
      cmp = (cmp_t)monomial_cmp_grlex;
  }
  qsort(pts->pt,pts->npt,sizeof(tuple),cmp);
}

PREFIX uma main_algo_ex(struct points *pts,int order,int algo,
                        unsigned long mem)
{
  switch(algo) {
    case IMUNAL_NAIVE_ALGO: return nalgo(pts,order);
    case IMUNAL_GREEDY_ALGO: return galgo(pts,order);
    case IMUNAL_NOSWAP_ALGO: return salgo(pts,order,mem);
    default: return main_algo(pts,order);
  }
}

PREFIX uma main_algo(struct points *pts,int order) {
  unsigned long ram;
  uma uma_ram,D;

  /* get parameters */
  D = pts->npt;
  ram = get_ram();
  uma_ram = ram;
  
  /* for debugging */
  /*fprintf(stderr,"RAM = %lu\n",ram);*/
  
  /* unable to determine amount of RAM => naive algo */
  if ( ram == get_ram_error ) return nalgo(pts,order);
  
  /* enough memory for one bit per char => naive algo */
  if ( nmat_formula(D,0) * (uma)sizeof(gmat_coeff) <
       3 * (uma_ram / 4) )
  {
    return nalgo(pts,order);
  }
  
  /* enough memory for 4 bits per char => greedy version */
  if ( gmat_formula(D) * (uma)sizeof(gmat_coeff) <
       3 * (uma_ram / 4) )
  {
    return galgo(pts,order);
  }
  
  /* not enough memory => siouks algo */
  return salgo(pts,order,3 * (uma_ram / 4));
}

