/*
 * This file is part of imunal
 * Copyright 2013  Guillaume Quintin, Olivier Ruatta, Philippe Gaborit
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#define _POSIX_C_SOURCE 200112L

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>

#define MB(a)       ((a) * 1024 * 1014)

#define PROG_NAME   "./imunal" /* program to be exec */
#define LIMIT_MEM   10         /* limit in MB */
#define LIMIT_TIME  5          /* limit in seconds */

void end(const char *msg) {
  fputs("wrapper: ",stderr);
  fputs(msg,stderr);
  fputc('\n',stderr);
  fflush(stderr);
  exit(-1);
}

void signal_handler(int s) {
  (void)s;
}

void set_signal_handler(int s) {
  struct sigaction act;
  act.sa_handler = signal_handler;
  act.sa_flags = 0;
  sigemptyset(&act.sa_mask);
  sigaddset(&act.sa_mask,s);
  if ( sigaction(s,&act,NULL) < 0 ) {
    switch(s) {
      case SIGCHLD:
        end("cannot set handler for SIGCHLD");
      case SIGINT:
        end("cannot set handler for SIGINT");
      case SIGTERM:
        end("cannot set handler for SIGTERM");
      default:
        end("cannot set handler for unknown signal");
    }
  }
}

int main(int argc,char **argv) {
  struct rlimit mem;
  struct timespec tv;
  pid_t child;
  int ret;

  /* limits */
  mem.rlim_cur = MB(LIMIT_MEM);
  mem.rlim_max = MB(LIMIT_MEM);
  tv.tv_sec = LIMIT_TIME;
  tv.tv_nsec = 0;
  
  /* set signals */
  set_signal_handler(SIGCHLD);
  set_signal_handler(SIGINT);
  set_signal_handler(SIGTERM);

  /* go for it */
  child = fork();
  if ( child == (pid_t)-1 ) end("cannot fork");
  
  /* in the child */
  if ( child == 0 ) {
    char **child_argv;
    int i;

    if ( setrlimit(RLIMIT_AS,&mem) ) exit(-1);
    if ( signal(SIGCHLD,SIG_DFL) == SIG_ERR ) exit(-1);
    if ( signal(SIGINT,SIG_DFL) == SIG_ERR ) exit(-1);
    if ( signal(SIGTERM,SIG_DFL) == SIG_ERR ) exit(-1);
    child_argv = malloc((argc + 1) * sizeof(char*));
    if ( !child_argv ) exit(-1);
    child_argv[0] = PROG_NAME;
    for( i = 1 ; i < argc ; i++ ) {
      child_argv[i] = argv[i];
    }
    child_argv[argc] = NULL;
    errno = 0;
    execvp(PROG_NAME,child_argv);
    exit(-1);
  }

  /* in the parent */
  ret = 0;
  errno = 0;
  nanosleep(&tv,NULL);
  if ( errno == EINTR ) ret = -1;
  kill(child,SIGKILL);
  waitpid(child,NULL,0);
  
  /* always return happily */
  return ret;
}

